<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\UserAuth;
use App\Models\Office;
use App\Models\RoleMaster;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use App\Traits\ApiResponseTrait;
use Illuminate\Support\Facades\Password;
use Illuminate\Support\Facades\Mail;
use App\Mail\WelcomeUserMail;
use App\Models\UserDeleteLog;
use Illuminate\Support\Str;

class UserController extends Controller
{
    use ApiResponseTrait;


    private function generateUserCode()
    {
        do {
            // Generate 5-character alphanumeric code
            $code = strtoupper(Str::random(5));

            // Check if already exists
            $exists = User::where('user_code', $code)->exists();
        } while ($exists); // If exists → repeat

        return $code;
    }



    public function createUser(Request $request)
    {
        $data = $request->validate([
            'first_name' => 'required|string|max:150',
            'last_name'  => 'nullable|string|max:150',
            'email'      => 'required|email|unique:user_auth,email',
            'role_id'    => 'required|exists:user_role_master,id',
            'office_id'  => 'required|exists:office,id',
            'phone'                     => 'nullable|string|max:25',
            'contact'                   => 'nullable|string|max:25',
            'address'                   => 'nullable|string',
            'city_id'        => 'required|exists:city_master,id',
            'state_id'       => 'required|exists:state_master,id',
            'pincode'        => 'required|integer',
            'photo'                     => 'nullable|string',
            'joining_date'             => 'nullable|date',
            'emergency_contact_name'   => 'nullable|string|max:150',
            'emergency_contact_phone'  => 'nullable|string|max:25',
            'emergency_contact_mobile' => 'nullable|string|max:25',
            'emergency_contact_address' => 'nullable|string',
            'headers'                  => 'nullable|string',
            'register_ip'              => 'nullable|string|max:200'
        ]);

        $office = Office::where('id', $data['office_id'])
            ->where('is_deleted', 0)
            ->first();

        if (!$office) {
            return $this->errorResponse('Office not found or inactive', null, 404);
        }
        $userCode = $this->generateUserCode();

        // Generate temporary random password
        $plainPassword = Str::password(12);
        // CREATE LOGIN RECORD
        $auth = UserAuth::create([
            'email'      => $data['email'],
            'username'   => $data['email'],
            'password'   => Hash::make($plainPassword),
            'is_active'  => 1
        ]);

        // AUTO-GENERATE USER CODE

        $ip = $request->ip();
        $headersJson = json_encode($request->headers->all());

        $user_type = 0;
        if($data['office_id'] > 1){
            $user_type = 1;
        }

        // CREATE PROFILE RECORD
        $user = User::create([
            'user_auth_id' => $auth->id,
            'user_code'    => $userCode,
            'first_name'   => $data['first_name'],
            'last_name'    => $data['last_name'],
            'email'        => $data['email'],
            'office_id'    => $data['office_id'],
            'role_id'      => $data['role_id'],
            'user_type'    => $user_type,
            'is_active'    => 1,
            'phone'        => $data['phone'] ?? null,
            'contact'      => $data['contact'] ?? null,
            'address'      => $data['address'] ?? null,
            'city_id'        => $data['city_id'],
            'state_id'       => $data['state_id'],
            'pincode'        => $data['pincode'],
            'photo'        => $data['photo'] ?? null,
            'joining_date' => $data['joining_date'] ?? null,
            'emergency_contact_name'   => $data['emergency_contact_name'] ?? null,
            'emergency_contact_phone'  => $data['emergency_contact_phone'] ?? null,
            'emergency_contact_mobile' => $data['emergency_contact_mobile'] ?? null,
            'emergency_contact_address' => $data['emergency_contact_address'] ?? null,

            'headers'      => $data['headers'] ?? $headersJson,
            'register_ip'  => $data['register_ip'] ?? $ip,
        ]);


        $token = Str::random(64);

        $auth->update([
            'reset_token'       => $token,
            'token_expires_on'  => now()->addMinutes(60),
        ]);


        // Send Welcome Email
        Mail::to($data['email'])->send(new WelcomeUserMail($user, $token));


        return $this->successResponse('User created successfully, welcome email sent');
    }




    public function listUsers($id)
    {
        // Fetch the requesting user
        $user = User::where('id', $id)
            ->where('is_deleted', 0)
            ->first();

        if (!$user) {
            return $this->errorResponse('User not found', null, 404);
        }

        $userType = $user->user_type;

        $query = User::where('is_deleted', 0);

        if ($userType == 0) {
            $query->whereIn('user_type', [0, 1]);     //HO

        } elseif ($userType == 1) {
            $query->where('user_type', 1);          //BO
            $query->where('office_id', $user->office_id);          //BO

        } elseif ($userType == 2) {
            $query->whereIn('user_type', [2, 3]);   //CM

        } elseif ($userType == 3) {
            $query->where('user_type', 3);          //CB
            $query->where('client_branch_id', $user->client_branch_id);
        }

        $query->with(['role']);

        $query->with([
            'office' => fn($q) =>
            $q->where('is_deleted', 0)
                ->orderBy('is_head_office', 'ASC')
        ]);

        $query->with([
            'clientBranch' => fn($q) =>
            $q->where('is_deleted', 0)

        ]);

        // // Final ordering
        // $query->orderBy('id', 'DESC');

        // Fetch results
        $users = $query->get();

        return $this->successResponse('Users fetched successfully', $users);
    }



    public function listOrgUsers()
    {
        $users = User::where('is_deleted', 0)
            ->where('user_type', 1)
            ->with(['role'])
            ->with(['office' => fn($q) => $q->where('is_deleted', 0)])
            ->orderBy('id', 'DESC')
            ->get();

        return $this->successResponse('Users fetched successfully', $users);
    }


    /**
     * GET SINGLE USER DETAILS
     */
    public function getUser($id)
    {
        $user = User::where('id', $id)
            ->where('is_deleted', 0)
            ->with(['role'])
            ->with(['office' => fn($q) => $q->where('is_deleted', 0)])
            ->with(['clientBranch' => fn($q) => $q->where('is_deleted', 0)])
            ->first();

        if (!$user) {
            return $this->errorResponse('User not found', null, 404);
        }

        return $this->successResponse('User fetched successfully', $user);
    }


    /**
     * UPDATE USER DETAILS
     */
    public function updateUser(Request $request, $id)
    {
        $user = User::where('id', $id)
            ->where('is_deleted', 0)
            ->first();

        if (!$user) {
            return $this->errorResponse('User not found', null, 404);
        }

        $data = $request->validate([
            'first_name' => 'nullable|string|max:150',
            'last_name'  => 'nullable|string|max:150',
            'role_id'    => 'nullable|exists:user_role_master,id',
            'office_id'  => 'nullable|exists:office,id',
            'email'      => 'nullable|email|unique:user_auth,email,' . $user->auth->id,
            'phone'                     => 'nullable|string|max:25',
            'contact'                   => 'nullable|string|max:25',
            'address'                   => 'nullable|string',
            'city_id'        => 'required|exists:city_master,id',
            'state_id'       => 'required|exists:state_master,id',
            'pincode'        => 'required|integer',
            'photo'                     => 'nullable|string',
            'joining_date'             => 'nullable|date',
            'relived_on'               => 'nullable|date',
            'emergency_contact_name'   => 'nullable|string|max:150',
            'emergency_contact_phone'  => 'nullable|string|max:25',
            'emergency_contact_mobile' => 'nullable|string|max:25',
            'emergency_contact_address' => 'nullable|string',
            'is_active'                 => 'required|integer',
        ]);

        // If email changes → update both tables
        if (isset($data['email'])) {
            $user->auth->update(['email' => $data['email']]);
        }
        $user->update($data);

        return $this->successResponse('User updated successfully');
    }


    /**
     * CHANGE USER PASSWORD
     */
    public function changePassword(Request $request, $id)
    {
        $user = User::where('id', $id)
            ->where('is_deleted', 0)
            ->first();

        if (!$user) {
            return $this->errorResponse('User not found', null, 404);
        }

        $request->validate([
            'password' => 'required|min:6|confirmed'
        ]);

        $user->auth->update([
            'password' => Hash::make($request->password)
        ]);

        return $this->successResponse('Password updated successfully');
    }


    /**
     * DELETE USER (SOFT DELETE)
     */
    public function deleteUser($id, Request $request)
    {
        $user = User::where('id', $id)
            ->where('is_deleted', 0)
            ->first();

        if (!$user) {
            return $this->errorResponse('User not found', null, 404);
        }
        if ($user->role_id == 1 || $user->role_id == 2) {
            return $this->errorResponse('Admin user cannot be deleted', null, 404);
        }

        UserDeleteLog::create([
            'table_id' => $id,
            'table_name'  => 'users',
            'reason' => $request['reason'],
        ]);

        $user->update(['is_deleted' => 1]);

        return $this->successResponse('User deleted successfully');
    }
}
