<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Team;
use App\Models\TeamUsersMapping;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use App\Traits\ApiResponseTrait;

class TeamController extends Controller
{
    use ApiResponseTrait;

    /**
     * List all teams
     */
   public function listTeams()
{
    try {
        $teams = Team::where('is_deleted', 0)
            ->with(['team_members.user:id,first_name,last_name,email'])
            ->orderBy('created_on', 'DESC')
            ->get()
            ->map(function ($team) {
                return [
                    'id' => $team->id,
                    'team_name' => $team->team_name,
                    'is_active' => (bool) $team->is_active,
                    'member_count' => $team->team_members->count(),
                    'created_on' => $team->created_on,
                    'updated_on' => $team->updated_on,
                    'members' => $team->team_members->map(function ($member) {
                        $user = $member->user;

                        return [
                            'id' => $member->id,
                            'user_id' => $member->user_id,
                            'user_name' => trim($user->first_name . ' ' . ($user->last_name ?? '')),
                            'user_email' => $user->email ?? '',
                            'created_on' => $member->created_on,
                        ];
                    }),
                ];
            });

        return $this->successResponse('Teams list fetched successfully', $teams);
    } catch (\Exception $e) {
        return $this->errorResponse('Failed to fetch teams', $e->getMessage(), 500);
    }
}


    /**
     * Get single team details
     */
    public function getTeam($id)
    {
        try {
            $team = Team::where('id', $id)
                ->where('is_deleted', 0)
                ->with(['team_members.user:id,first_name,last_name,email'])
                ->first();

            if (!$team) {
                return $this->errorResponse('Team not found', null, 404);
            }

            return $this->successResponse('Team details fetched successfully', [
                'id' => $team->id,
                'team_name' => $team->team_name,
                'is_active' => (bool) $team->is_active,
                'created_on' => $team->created_on,
                'updated_on' => $team->updated_on,
                'members' => $team->team_members->map(function ($member) {
                    $user = $member->user;
                    return [
                        'id' => $member->id,
                        'user_id' => $member->user_id,
                        'user_name' => trim($user->first_name . ' ' . ($user->last_name ?? '')),
                        'user_email' => $user->email ?? '',
                        'is_active' => (bool) $member->is_active,
                        'created_on' => $member->created_on,
                        'updated_on' => $member->updated_on,
                    ];
                }),
            ]);
        } catch (\Exception $e) {
            return $this->errorResponse('Failed to fetch team', $e->getMessage(), 500);
        }
    }

    /**
     * Create team
     */
    public function createTeam(Request $request)
    {
        $data = $request->validate([
            'team_name' => 'required|string|max:255|unique:team,team_name',
            'is_active' => 'nullable|boolean',
        ]);

        DB::beginTransaction();

        try {
            $team = Team::create([
                'team_name' => $data['team_name'],
                'is_active' => $data['is_active'] ?? 1,
                'is_deleted' => 0,
            ]);

            DB::commit();
            return $this->successResponse('Team created successfully', $team);
        } catch (\Exception $e) {
            DB::rollBack();
            return $this->errorResponse('Failed to create team', $e->getMessage(), 500);
        }
    }

    /**
     * Update team
     */
    public function updateTeam(Request $request, $id)
    {
        $team = Team::where('id', $id)->where('is_deleted', 0)->first();

        if (!$team) {
            return $this->errorResponse('Team not found', null, 404);
        }

        $data = $request->validate([
            'team_name' => 'required|string|max:255|unique:team,team_name,' . $id,
            'is_active' => 'required|boolean',
        ]);

        DB::beginTransaction();

        try {
            $team->update($data);
            DB::commit();
            return $this->successResponse('Team updated successfully', $team);
        } catch (\Exception $e) {
            DB::rollBack();
            return $this->errorResponse('Failed to update team', $e->getMessage(), 500);
        }
    }

    /**
     * Delete team (soft delete)
     */
    public function deleteTeam($id)
    {
        $team = Team::where('id', $id)->where('is_deleted', 0)->first();

        if (!$team) {
            return $this->errorResponse('Team not found', null, 404);
        }

        $activeMembers = TeamUsersMapping::where('team_id', $id)
            ->where('is_active', 1)
            ->where('is_deleted', 0)
            ->count();

        if ($activeMembers > 0) {
            return $this->errorResponse(
                'Cannot delete team with active members',
                ['active_members_count' => $activeMembers],
                400
            );
        }

        DB::beginTransaction();

        try {
            $team->update(['is_deleted' => 1]);

            TeamUsersMapping::where('team_id', $id)
                ->update(['is_deleted' => 1]);

            DB::commit();
            return $this->successResponse('Team deleted successfully');
        } catch (\Exception $e) {
            DB::rollBack();
            return $this->errorResponse('Failed to delete team', $e->getMessage(), 500);
        }
    }

    /**
     * Get team members
     */
  public function getTeamMembers($teamId)
{
    try {
        $team = Team::where('id', $teamId)
            ->where('is_deleted', 0)
            ->first();

        if (!$team) {
            return $this->errorResponse('Team not found', null, 404);
        }

        $members = TeamUsersMapping::where('team_id', $teamId)
            ->where('is_deleted', 0)
            ->with([
                'user:id,first_name,last_name,email,role_id,office_id',
                'user.role:id,name',
                'user.office:id,name'
            ])
            ->get()
            ->map(function ($member) {
                $user = $member->user;

                return [
                    'id' => $member->id,
                    'user_id' => $member->user_id,
                    'user_name' => $user
                        ? trim($user->first_name . ' ' . ($user->last_name ?? ''))
                        : 'Unknown',
                    'user_email' => $user->email ?? '',
                    'is_active' => (bool) $member->is_active,
                    'created_on' => $member->created_on,
                    'updated_on' => $member->updated_on,

                    // Role
                    'role_id' => $user?->role?->id,
                    'role_name' => $user?->role?->name,

                    // Office
                    'office_id' => $user?->office?->id,
                    'office_name' => $user?->office?->name,
                ];
            });

        return $this->successResponse('Team members fetched successfully', [
            'team_id' => $team->id,
            'team_name' => $team->team_name,
            'members' => $members,
            'total_members' => $members->count(),
        ]);
    } catch (\Exception $e) {
        return $this->errorResponse('Failed to fetch team members', $e->getMessage(), 500);
    }
}


    /**
     * Get available users for team
     */
   public function getAvailableUsers($teamId)
{
    try {
        $existingUserIds = TeamUsersMapping::where('team_id', $teamId)
            ->where('is_deleted', 0)
            ->pluck('user_id');

        $users = User::whereNotIn('id', $existingUserIds)
            ->where('is_active', 1)
            ->whereIn('user_type', [0, 1])
            ->with(['role:id,name', 'office:id,name'])
            ->get()
            ->map(function ($user) {
                return [
                    'id' => $user->id,
                    'name' => trim($user->first_name . ' ' . ($user->last_name ?? '')),
                    'email' => $user->email,
                    'user_code' => $user->user_code,
                    'role_id' => $user->role->id ?? null,
                    'role_name' => $user->role->name ?? null,
                    'office_id' => $user->office->id ?? null,
                    'office_name' => $user->office->name ?? null,
                ];
            });

        return $this->successResponse('Available users fetched successfully', $users);
    } catch (\Exception $e) {
        return $this->errorResponse('Failed to fetch available users', $e->getMessage(), 500);
    }
}


    /**
     * Add user to team
     */
    public function addUserToTeam(Request $request, $teamId)
    {
        $data = $request->validate([
            'user_id' => 'required|exists:users,id',
            'is_active' => 'nullable|boolean',
        ]);

        DB::beginTransaction();

        try {
            $mapping = TeamUsersMapping::where('team_id', $teamId)
                ->where('user_id', $data['user_id'])
                ->where('is_deleted', 0)
                ->first();

            if ($mapping) {
                if ($mapping->is_active) {
                    return $this->errorResponse('User already in team', null, 400);
                }
                $mapping->update(['is_active' => 1]);
                DB::commit();
                return $this->successResponse('User reactivated in team', $mapping);
            }

            $member = TeamUsersMapping::create([
                'team_id' => $teamId,
                'user_id' => $data['user_id'],
                'is_active' => $data['is_active'] ?? 1,
                'is_deleted' => 0,
            ]);

            DB::commit();
            return $this->successResponse('User added to team', $member);
        } catch (\Exception $e) {
            DB::rollBack();
            return $this->errorResponse('Failed to add user', $e->getMessage(), 500);
        }
    }

    /**
     * Remove user from team
     */
    public function removeUserFromTeam($teamId, $memberId)
    {
        DB::beginTransaction();

        try {
            $member = TeamUsersMapping::where('id', $memberId)
                ->where('team_id', $teamId)
                // ->where('is_deleted', 0)
                ->first();

            if (!$member) {
                return $this->errorResponse('Team member not found', null, 404);
            }

            // $member->update(['is_deleted' => 1]);
            $member->delete();
            DB::commit();

            return $this->successResponse('User removed from team');
        } catch (\Exception $e) {
            DB::rollBack();
            return $this->errorResponse('Failed to remove user', $e->getMessage(), 500);
        }
    }

    /**
     * Toggle user status in team
     */
    public function toggleUserStatus($teamId, $memberId)
    {
        $member = TeamUsersMapping::where('id', $memberId)
            ->where('team_id', $teamId)
            ->where('is_deleted', 0)
            ->first();

        if (!$member) {
            return $this->errorResponse('Team member not found', null, 404);
        }

        $member->update(['is_active' => !$member->is_active]);

        return $this->successResponse('User status updated', $member);
    }

    /**
     * Get teams for user
     */
    public function getUserTeams($userId)
    {
        try {
            $teams = TeamUsersMapping::where('user_id', $userId)
                ->where('is_deleted', 0)
                ->where('is_active', 1)
                ->with('team')
                ->get()
                ->map(function ($mapping) {
                    return [
                        'team_id' => $mapping->team_id,
                        'team_name' => $mapping->team->team_name ?? 'Unknown',
                        'is_active' => (bool) $mapping->team->is_active,
                        'added_on' => $mapping->created_on,
                        'user_role_in_team' => 'member',
                    ];
                });

            return $this->successResponse('User teams fetched successfully', $teams);
        } catch (\Exception $e) {
            return $this->errorResponse('Failed to fetch user teams', $e->getMessage(), 500);
        }
    }

    /**
     * Search teams
     */
    public function searchTeams(Request $request)
    {
        try {
            $search = $request->get('search', '');

            $teams = Team::where('is_deleted', 0)
                ->where('team_name', 'like', "%{$search}%")
                ->withCount([
                    'team_members as member_count' => function ($q) {
                        $q->where('is_active', 1)->where('is_deleted', 0);
                    }
                ])
                ->orderBy('created_on', 'DESC')
                ->get();

            return $this->successResponse('Teams search completed', [
                'search_term' => $search,
                'teams' => $teams,
                'total_results' => $teams->count(),
            ]);
        } catch (\Exception $e) {
            return $this->errorResponse('Failed to search teams', $e->getMessage(), 500);
        }
    }
}
