<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Traits\ApiResponseTrait;
use App\Models\User;
use App\Models\UserAuth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Password;
use Illuminate\Support\Facades\Mail;
use App\Mail\ClientWelcomeMail;
use App\Models\ClientMaster;
use Illuminate\Support\Facades\DB;
use App\Models\ClientBranch;
use App\Models\ClientBranchSystemUsermaping;
use App\Models\OfficeClientBranchMapping;
use Illuminate\Support\Str;
use App\Models\UserDeleteLog;


class ClientController extends Controller
{
    use ApiResponseTrait;


    private function generateClientCode()
    {
        do {
            // Generate 5-character alphanumeric code
            $code = strtoupper(Str::random(5));

            // Check if already exists
            $exists = ClientMaster::where('client_code', $code)->exists();
        } while ($exists); // If exists → repeat

        return $code;
    }

    private function generateClientBranchCode()
    {
        do {
            // Generate 5-character alphanumeric code
            $code = strtoupper(Str::random(5));

            // Check if already exists
            $exists = ClientBranch::where('branch_code', $code)->exists();
        } while ($exists); // If exists → repeat

        return $code;
    }

    private function generateClientUserCode()
    {
        do {
            // Generate 5-character alphanumeric code
            $code = strtoupper(Str::random(5));

            // Check if already exists
            $exists = User::where('user_code', $code)->exists();
        } while ($exists); // If exists → repeat

        return $code;
    }

    private function generateNumericUsername()
    {
        do {
            // random 6-digit number
            $username = random_int(100000, 999999);
        } while (\App\Models\UserAuth::where('username', $username)->exists());

        return $username;
    }


    public function createClient(Request $request)
    {
        $data = $request->validate([
            'client_name'        => 'required|string|max:200',
            'client_bussiness_name'        => 'nullable|string|max:150',
            'department_id'      => 'required|exists:client_department_master,id',
            'source_id'          => 'required|exists:client_source_master,id',
            'classification_id'  => 'required|exists:client_classification_master,id',
            'address'            => 'required|string',
            'state_id'           => 'required|exists:state_master,id',
            'city_id'            => 'required|exists:city_master,id',
            'pincode'              => 'required|string|max:10',
            'cost'               => 'nullable|numeric',
            'effective_month'    => 'nullable|date',
            'email'              => 'required|email|unique:client_master,email',
            // 'contact_no'         => 'nullable|string',
            'logo'               => 'nullable|string',

        ]);

        // AUTO CLIENT CODE
        $clientCode = $this->generateClientCode();

        $client = ClientMaster::create(array_merge($data, [
            'client_code' => $clientCode,
            'is_active'   => 1,
            'is_deleted'  => 0,
        ]));


        if (!empty($data['email'])) {

            $tempPassword = Str::password(12);

            $ip = $request->ip();
            $headersJson = json_encode($request->headers->all());

            $auth = UserAuth::create([
                'email'     => $data['email'],
                // 'username'  => $this->generateNumericUsername(),
                'username'  => $data['email'],
                'password'  => Hash::make($tempPassword),
                'is_active' => 1,
            ]);
            $clientUserCode = $this->generateClientUserCode();
            $user = User::create([
                'user_auth_id' => $auth->id,
                'user_code'    => $clientUserCode,
                'first_name'   => $data['client_spoc_name'] ?? 'Client Admin',
                'email'        => $data['email'],
                'address'      => $data['address'] ?? null,
                'city_id'        => $data['city_id'],
                'state_id'       => $data['state_id'],
                'pincode'        => $data['pincode'],
                'client_id'    => $client->id,
                'role_id'      => 10,           // Client master super admin
                'user_type'    => 2,
                'is_active'    => 1,
                'headers'      => $data['headers'] ?? $headersJson,
                'register_ip'  => $data['register_ip'] ?? $ip,
            ]);

            // Create password token
            $token = Str::random(64);

            // Send welcome email
            Mail::to($data['email'])->send(new ClientWelcomeMail($user, $token));
        }

        return $this->successResponse('Client created successfully', [
            'client_details' => $client,

        ]);
    }


    /**
     * CREATE CLIENT Branch
     */
    public function createClientBranch(Request $request)
    {
        $data = $request->validate([
            'client_id'          => 'required|exists:client_master,id',
            'branch_name'        => 'required|string|max:200',
            'email'              => 'required|email|unique:client_branch,email',
            'address'            => 'nullable|string',
            'state_id'           => 'nullable|exists:state_master,id',
            'city_id'            => 'nullable|exists:city_master,id',
            'pincode'            => 'nullable|string|max:10',

            'spoc_name'          => 'nullable|string|max:150',
            'spoc_email'         => 'nullable|email',
            'spoc_contact'       => 'nullable|string|max:25',

            'escalation_name'    => 'nullable|string|max:150',
            'escalation_email'   => 'nullable|email',
            'escalation_contact' => 'nullable|string|max:25',

            'effective_month'    => 'nullable|date',

            'team_administration'               => 'required|array|min:1',

            'team_administration.*.team_id'   => 'required|exists:team,id',
            'team_administration.*.office_id'   => 'required|exists:office,id',
            'team_administration.*.user_id'     => 'required|exists:users,id',
            'team_administration.*.role_id'     => 'required|exists:user_role_master,id',
             'team_administration.*.is_active'   => 'required|integer|in:0,1',


            'office_id'          => 'required|exists:office,id',
        ]);

        DB::beginTransaction();

        try {
            /* ---------------------------------
         | Generate Branch Code
         --------------------------------- */
            $branchCode = $this->generateClientBranchCode();

            /* ---------------------------------
         | Create Client Branch
         --------------------------------- */
            $branch = ClientBranch::create([
                'client_id'          => $data['client_id'],
                'branch_name'        => $data['branch_name'],
                'email'              => $data['email'],
                'address'            => $data['address'] ?? null,
                'state_id'           => $data['state_id'] ?? null,
                'city_id'            => $data['city_id'] ?? null,
                'pincode'            => $data['pincode'] ?? null,

                'spoc_name'          => $data['spoc_name'] ?? null,
                'spoc_email'         => $data['spoc_email'] ?? null,
                'spoc_contact'       => $data['spoc_contact'] ?? null,

                'escalation_name'    => $data['escalation_name'] ?? null,
                'escalation_email'   => $data['escalation_email'] ?? null,
                'escalation_contact' => $data['escalation_contact'] ?? null,

                'effective_month'    => $data['effective_month'] ?? null,
                'branch_code'        => $branchCode,
                'is_active'          => 1,
                'is_deleted'         => 0,
            ]);

            $branchId = $branch->id;
            $clientId = $data['client_id'];

            /* ---------------------------------
         | Office ↔ Branch Mapping
         --------------------------------- */
            OfficeClientBranchMapping::create([
                'office_id'        => $data['office_id'],
                'client_branch_id' => $branchId,
            ]);

            /* ---------------------------------
         | System User Mappings
         --------------------------------- */
            foreach ($data['team_administration'] as $row) {
                ClientBranchSystemUsermaping::create([
                    'team_id'  => $row['team_id'],
                    'client_id'  => $clientId,
                    'branch_id'  => $branchId,
                    'office_id'  => $row['office_id'],
                    'role_id'  => $row['role_id'],
                    'user_id'    => $row['user_id'],
                    'is_active'  => $row['is_active'],
                    'is_deleted' => 0,
                ]);
            }

            /* ---------------------------------
         | Create Branch Admin Login (SPOC)
         --------------------------------- */
            $user = null;
            $token = null;

            if (!empty($data['spoc_email'])) {

                $tempPassword = Str::password(12);
                $ip = $request->ip();
                $headersJson = json_encode($request->headers->all());

                $auth = UserAuth::create([
                    'email'     => $data['spoc_email'],
                    'username'  => $data['spoc_email'],
                    'password'  => Hash::make($tempPassword),
                    'is_active' => 1,
                ]);

                $branchUserCode = $this->generateClientUserCode();

                $user = User::create([
                    'user_auth_id'     => $auth->id,
                    'user_code'        => $branchUserCode,
                    'first_name'       => $data['spoc_name'],
                    'email'            => $data['spoc_email'],
                    'client_branch_id' => $branchId,
                    'role_id'          => 12,                   //client Branch Admin
                    'user_type'        => 3,
                    'is_active'        => 1,
                    'headers'          => $headersJson,
                    'register_ip'      => $ip,
                ]);

                $token = Str::random(64);
            }

            DB::commit();

            /* ---------------------------------
         | Send Welcome Email AFTER COMMIT
         --------------------------------- */
            if ($user && $token) {
                Mail::to($data['spoc_email'])->send(
                    new ClientWelcomeMail($user, $token)
                );
            }

            return $this->successResponse(
                'Client branch created successfully',
                ['branch' => $branch]
            );
        } catch (\Exception $ex) {
            DB::rollBack();

            return $this->errorResponse(
                'Error creating branch',
                $ex->getMessage(),
                500
            );
        }
    }




    public function listClients()
    {
        $clients = ClientMaster::where('is_deleted', 0)
            ->with(['city_master', 'state_master'])
            ->orderBy('client_name')
            ->get();

        return $this->successResponse('Client list fetched successfully', $clients);
    }

    public function listClientBranch($id)
    {
        $client = ClientMaster::where('id', $id)
            ->where('is_deleted', 0)
            ->first();
        if (!$client) {
            return $this->errorResponse('Client not found', null, 404);
        }

        $clients = ClientBranch::where('is_deleted', 0)
            ->where('client_id', $client->id)
            ->with(['city_master', 'state_master'])
            ->orderBy('branch_name')
            ->get();

        return $this->successResponse('Client branch list fetched successfully', $clients);
    }


    public function getClient($id)
    {
        $client = ClientMaster::where('id', $id)
            ->where('is_deleted', 0)
            ->with([
                'department',
                'source',
                'classification',
                'city_master',
                'state_master',

                // load branches WITH filter + relations
                'branches' => fn($q) =>
                $q->where('is_deleted', 0)
                    ->with([
                        'city_master',
                        'state_master',

                        'client_branch_users' => fn($u) =>
                        $u->where('is_deleted', 0)
                            ->with(['city_master', 'state_master'])
                    ])
            ])
            ->first();

        if (!$client) {
            return $this->errorResponse('Client not found', null, 404);
        }

        return $this->successResponse('Client details fetched successfully', $client);
    }

   public function getClientBranch($id)
{
    $clientBranch = ClientBranch::where('id', $id)
        ->where('is_deleted', 0)
        ->with([
            'branch_system_user_mapping' => function($q) {
                return $q->where('is_deleted', 0)
                    ->with([
                        'user:id,first_name,last_name,email,user_code',
                        'office:id,name',
                        'role:id,name',
                        'team:id,team_name' // Add this if you have team relationship
                    ]);
            },
            'office_client_branch_mapping' => function($q) {
                return $q->select('id', 'client_branch_id', 'office_id');
            },
        ])
        ->first();

    if (!$clientBranch) {
        return $this->errorResponse('Client branch not found', null, 404);
    }

    return $this->successResponse(
        'Client branch details fetched successfully',
        $clientBranch
    );
}


    /**
     * UPDATE CLIENT
     */
    public function updateClient(Request $request, $id)
    {
        $client = ClientMaster::where('id', $id)
            ->where('is_deleted', 0)
            ->first();

        if (!$client) {
            return $this->errorResponse('Client not found', null, 404);
        }

        $data = $request->validate([
            'client_name'        => 'required|string|max:200',
            'client_bussiness_name'        => 'nullable|string|max:150',
            'department_id'      => 'required|exists:client_department_master,id',
            'source_id'          => 'required|exists:client_source_master,id',
            'classification_id'  => 'required|exists:client_classification_master,id',
            'address'            => 'nullable|string',
            'state_id'           => 'nullable|exists:state_master,id',
            'city_id'            => 'nullable|exists:city_master,id',
            'cost'               => 'nullable|numeric',
            'effective_month'    => 'nullable|date',
            'email'              => 'required|email|unique:client_master,email,' . $client->id,
            'logo'               => 'nullable|string',
            'pincode'              => 'nullable|string|max:10',
            'is_active'                    => 'required|integer',
        ]);

        $client->update($data);

        return $this->successResponse('Client updated successfully', $client);
    }

    public function updateClientBranch(Request $request, $id)
    {
        /* ---------------------------------
     | Fetch Branch
     --------------------------------- */
        $clientBranch = ClientBranch::where('id', $id)
            ->where('is_deleted', 0)
            ->first();

        if (!$clientBranch) {
            return $this->errorResponse('Client Branch not found', null, 404);
        }

        /* ---------------------------------
     | Validate Request
     --------------------------------- */
        $data = $request->validate([
            'branch_name'        => 'required|string|max:200',
            'email'              => 'required|email|unique:client_branch,email,' . $clientBranch->id,
            'address'            => 'nullable|string',
            'state_id'           => 'nullable|exists:state_master,id',
            'city_id'            => 'nullable|exists:city_master,id',
            'pincode'            => 'nullable|string|max:10',

            'spoc_name'          => 'nullable|string|max:150',
            'spoc_email'         => 'nullable|email',
            'spoc_contact'       => 'nullable|string|max:25',

            'escalation_name'    => 'nullable|string|max:150',
            'escalation_email'   => 'nullable|email',
            'escalation_contact' => 'nullable|string|max:25',

            'effective_month'    => 'nullable|date',

            // System users
            'team_administration'               => 'required|array|min:1',

            'team_administration.*.office_id'   => 'required|exists:office,id',
            'team_administration.*.user_id'     => 'required|exists:users,id',
            'team_administration.*.role_id'     => 'required|exists:user_role_master,id',
            'team_administration.*.is_active'   => 'required|integer|in:0,1',
            'team_administration.*.team_id'   => 'required|exists:team,id',
            'office_id'          => 'required|exists:office,id',
            'is_active'          => 'required|integer|in:0,1',
        ]);

        DB::beginTransaction();

        try {
            /* ---------------------------------
         | Update Branch
         --------------------------------- */
            $clientBranch->update([
                'branch_name'        => $data['branch_name'],
                'email'              => $data['email'],
                'address'            => $data['address'] ?? null,
                'state_id'           => $data['state_id'] ?? null,
                'city_id'            => $data['city_id'] ?? null,
                'pincode'            => $data['pincode'] ?? null,

                'spoc_name'          => $data['spoc_name'] ?? null,
                'spoc_email'         => $data['spoc_email'] ?? null,
                'spoc_contact'       => $data['spoc_contact'] ?? null,

                'escalation_name'    => $data['escalation_name'] ?? null,
                'escalation_email'   => $data['escalation_email'] ?? null,
                'escalation_contact' => $data['escalation_contact'] ?? null,

                'effective_month'    => $data['effective_month'] ?? null,
                'is_active'          => $data['is_active'],
            ]);

            $branchId = $clientBranch->id;
            $clientId = $clientBranch->client_id;

            /* ---------------------------------
         | Update Office ↔ Branch Mapping
         --------------------------------- */
            OfficeClientBranchMapping::where('client_branch_id', $branchId)->delete();

            OfficeClientBranchMapping::create([
                'office_id'        => $data['office_id'],
                'client_branch_id' => $branchId,
            ]);

            /* ---------------------------------
         | Reset System User Mappings
         --------------------------------- */
            ClientBranchSystemUsermaping::where('branch_id', $branchId)->delete();

           foreach ($data['team_administration'] as $row) {
                ClientBranchSystemUsermaping::create([

                    'client_id'  => $clientId,
                    'branch_id'  => $branchId,
                    'office_id'  => $row['office_id'],
                    'role_id'  => $row['role_id'],
                    'user_id'    => $row['user_id'],
                    'is_active'  => $row['is_active'],
                    'team_id'  => $row['team_id'],
                ]);
            }

            DB::commit();

            return $this->successResponse(
                'Client branch updated successfully',
                ['branch' => $clientBranch]
            );
        } catch (\Exception $ex) {
            DB::rollBack();

            return $this->errorResponse(
                'Error updating branch',
                $ex->getMessage(),
                500
            );
        }
    }



    /**
     * DELETE CLIENT (Soft delete)
     */
    public function deleteClient(Request $request, $id)
    {
        $client = ClientMaster::where('id', $id)->first();

        if (!$client) {
            return $this->errorResponse('Client not found', null, 404);
        }

        // Get all branches of this client
        $branches = ClientBranch::where('client_id', $client->id)->pluck('id');

        // Delete all client branch user mappings
        ClientBranchSystemUsermaping::whereIn('branch_id', $branches)->update(['is_deleted' => 1]);

        // Delete all users under all branches
        User::whereIn('client_branch_id', $branches)->update(['is_deleted' => 1]);

        // Delete client-level users also (client admin)
        User::where('client_id', $client->id)->update(['is_deleted' => 1]);

        // Delete all branches
        ClientBranch::where('client_id', $client->id)->update(['is_deleted' => 1]);

        UserDeleteLog::create([
            'table_id' => $id,
            'table_name'  => 'client_master',
            'reason' => $request['reason'],
        ]);

        // Delete client
        $client->update(['is_deleted' => 1]);

        return $this->successResponse('Client deleted successfully');
    }


    public function deleteClientBranch(Request $request, $id)
    {
        $clientBranch = ClientBranch::where('id', $id)->first();

        if (!$clientBranch) {
            return $this->errorResponse('Client Branch not found', null, 404);
        }

        // Delete user mappings for this branch
        ClientBranchSystemUsermaping::where('branch_id', $clientBranch->id)
            ->update(['is_deleted' => 1]);

        // Delete users under this branch
        User::where('client_branch_id', $clientBranch->id)
            ->update(['is_deleted' => 1]);

        UserDeleteLog::create([
            'table_id' => $id,
            'table_name'  => 'client_branch',
            'reason' => $request['reason'],
        ]);

        // Delete branch itself
        $clientBranch->update(['is_deleted' => 1]);

        return $this->successResponse('Client Branch deleted successfully');
    }



    /**
     * Create Branch User (user_type = 2)
     */
    public function createBranchUser(Request $request)
    {
        $data = $request->validate([
            'first_name'        => 'required|string|max:150',
            'last_name'         => 'nullable|string|max:150',
            'email'             => 'required|email|unique:user_auth,email',
            'client_branch_id'         => 'required|exists:client_branch,id',
            'role_id'           => 'required|exists:user_role_master,id',
            'phone'                     => 'nullable|string|max:25',
            'contact'                   => 'nullable|string|max:25',
            'address'                   => 'nullable|string',
            'state_id'           => 'required|exists:state_master,id',
            'city_id'            => 'required|exists:city_master,id',
            'pincode'   => 'nullable|string|max:10',
            'photo'                     => 'nullable|string',
            'joining_date'             => 'nullable|date',
            'relived_on'               => 'nullable|date',
            'emergency_contact_name'   => 'nullable|string|max:150',
            'emergency_contact_phone'  => 'nullable|string|max:25',
            'emergency_contact_mobile' => 'nullable|string|max:25',
            'emergency_contact_address' => 'nullable|string',
            'headers'                  => 'nullable|string',
            'register_ip'              => 'nullable|string|max:200'
        ]);

        // Generate temp password
        $plainPassword = Str::password(12);

        // Create Auth Record
        $auth = UserAuth::create([
            'email'      => $data['email'],
            'username'   => $data['email'],
            'password'   => Hash::make($plainPassword),
            'is_active'  => 1
        ]);

        // Auto user code
        $userCode = $this->generateClientUserCode();

        // Metadata
        $ip          = $request->ip();
        $headersJson = json_encode($request->headers->all());

        // Create User Profile
        $user = User::create([
            'user_auth_id'        => $auth->id,
            'user_code'           => $userCode,
            'first_name'          => $data['first_name'],
            'last_name'           => $data['last_name'] ?? null,
            'email'               => $data['email'],
            'client_branch_id'           => $data['client_branch_id'],
            'role_id'             => $data['role_id'],
            'user_type'           => 3,
            'is_active'           => 1,
            'phone'               => $data['phone'] ?? null,
            'contact'             => $data['contact'] ?? null,
            'address'             => $data['address'] ?? null,
            'state_id'           => $data['state_id'] ?? null,
            'city_id'            => $data['city_id'] ?? null,
            'pincode'            => $data['pincode'] ?? null,

            'photo'               => $data['photo'] ?? null,
            'joining_date'        => $data['joining_date'] ?? null,
            'relived_on'          => $data['relived_on'] ?? null,
            'emergency_contact_name'   => $data['emergency_contact_name'] ?? null,
            'emergency_contact_phone'  => $data['emergency_contact_phone'] ?? null,
            'emergency_contact_mobile' => $data['emergency_contact_mobile'] ?? null,
            'emergency_contact_address' => $data['emergency_contact_address'] ?? null,

            'headers'             => $data['headers'] ?? $headersJson,
            'register_ip'         => $data['register_ip'] ?? $ip,
        ]);

        // Reset token
        $token = Str::random(64);

        $auth->update([
            'reset_token'       => $token,
            'token_expires_on'  => now()->addMinutes(60),
        ]);

        // Send welcome email
        Mail::to($data['email'])->send(new ClientWelcomeMail($user, $token));

        return $this->successResponse('Branch user created successfully');
    }

    public function updateBranchUser(Request $request, $id)
    {
        // Fetch user
        $user = User::where('id', $id)
            ->where('is_deleted', 0)
            ->first();

        if (!$user) {
            return $this->errorResponse('Branch user not found', null, 404);
        }

        // Fetch auth record
        $auth = UserAuth::where('id', $user->user_auth_id)->first();

        if (!$auth) {
            return $this->errorResponse('User authentication record not found', null, 404);
        }

        // Validation
        $data = $request->validate([
            'first_name'        => 'required|string|max:150',
            'last_name'         => 'nullable|string|max:150',
            'email'             => 'required|email|unique:user_auth,email,' . $auth->id,
            'client_branch_id'      => 'required|exists:client_branch,id',
            'role_id'           => 'required|exists:user_role_master,id',

            'phone'                     => 'nullable|string|max:25',
            'contact'                   => 'nullable|string|max:25',
            'address'                   => 'nullable|string',
            'state_id'           => 'required|exists:state_master,id',
            'city_id'            => 'required|exists:city_master,id',
            'pincode'   => 'nullable|integer',
            'photo'                     => 'nullable|string',
            'joining_date'             => 'nullable|date',
            'relived_on'               => 'nullable|date',

            'emergency_contact_name'   => 'nullable|string|max:150',
            'emergency_contact_phone'  => 'nullable|string|max:25',
            'emergency_contact_mobile' => 'nullable|string|max:25',
            'emergency_contact_address' => 'nullable|string',

            'headers'                  => 'nullable|string',
            'register_ip'              => 'nullable|string|max:200',
            'is_active'                 => 'required|integer',
        ]);

        /** -----------------------------------------
         * UPDATE AUTH RECORD
         * ------------------------------------------*/
        $emailChanged = $auth->email !== $data['email'];

        $auth->update([
            'email'    => $data['email'],
            'username' => $data['email'],   // same rule as createBranchUser()
        ]);

        /** -----------------------------------------
         * UPDATE USER PROFILE RECORD
         * ------------------------------------------*/
        $ip          = $request->ip();
        $headersJson = json_encode($request->headers->all());

        $user->update([
            'first_name'          => $data['first_name'],
            'last_name'           => $data['last_name'] ?? null,
            'email'               => $data['email'],

            'client_branch_id'           => $data['client_branch_id'],  // branch reference
            'role_id'             => $data['role_id'],

            'phone'               => $data['phone'] ?? null,
            'contact'             => $data['contact'] ?? null,
            'address'             => $data['address'] ?? null,
            'state_id'           => $data['state_id'] ?? null,
            'city_id'            => $data['city_id'] ?? null,
            'pincode'            => $data['pincode'] ?? null,

            'photo'               => $data['photo'] ?? null,
            'joining_date'        => $data['joining_date'] ?? null,
            'relived_on'          => $data['relived_on'] ?? null,

            'emergency_contact_name'   => $data['emergency_contact_name'] ?? null,
            'emergency_contact_phone'  => $data['emergency_contact_phone'] ?? null,
            'emergency_contact_mobile' => $data['emergency_contact_mobile'] ?? null,
            'emergency_contact_address' => $data['emergency_contact_address'] ?? null,

            'headers'             => $data['headers'] ?? $headersJson,
            'register_ip'         => $data['register_ip'] ?? $ip,
            'is_active'             => $data['is_active']
        ]);

        /** --------------------------------------------------------
         * OPTIONALLY SEND NEW TOKEN IF EMAIL CHANGED
         * --------------------------------------------------------*/
        if ($emailChanged) {
            $token = Str::random(64);

            $auth->update([
                'reset_token'       => $token,
                'token_expires_on'  => now()->addMinutes(60),
            ]);

            Mail::to($data['email'])->send(new ClientWelcomeMail($user, $token));
        }

        return $this->successResponse('Branch user updated successfully', [
            'user_details' => $user
        ]);
    }
}
