<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Models\ComplianceMaster;
use App\Models\KeyAreasComplianceMaster;
use App\Models\KeyAreaSubCompliance;
use App\Traits\ApiResponseTrait;
use Illuminate\Support\Str;
use App\Models\UserDeleteLog;
use Illuminate\Support\Facades\DB;

class ComplianceController extends Controller
{
    use ApiResponseTrait;

    /**
     * List compliances
     */
    public function listCompliance(Request $request)
    {
        $query = ComplianceMaster::with([
            'state_master',
            'nature_compliance_master',
            'submission_authority_master'
        ])
            ->where('is_deleted', 0);

        if ($request->filled('state_id')) {
            $query->where('state_id', $request->state_id);
        }

        // if ($request->filled('compliance_month')) {
        //     $query->where('compliance_month', $request->compliance_month);
        // }

        $data = $query->orderBy('due_date')->get();

        return $this->successResponse('Compliance list fetched successfully', $data);
    }

    /**
     * Get single compliance
     */
    public function getCompliance($id)
    {
        $compliance = ComplianceMaster::with([
            'state_master',
            'nature_compliance_master',
            'submission_authority_master'
        ])
            ->where('id', $id)
            ->where('is_deleted', 0)
            ->first();

        if (!$compliance) {
            return $this->errorResponse('Compliance not found', null, 404);
        }

        return $this->successResponse('Compliance details fetched successfully', $compliance);
    }

    /**
     * Create compliance
     */
    public function createCompliance(Request $request)
    {
        $data = $request->validate([
            'compliance_month'          => 'required|date',
            'due_date'                  => 'required|date',
            'state_id'                  => 'required|exists:state_master,id',
            'statute_name'              => 'required|string|max:255',
            'form_name'                 => 'required|string|max:255',
            'nature_of_compliance_id'   => 'required|exists:nature_of_compliance_master,id',
            'submission_authority_id'   => 'required|exists:submission_authority_master,id',
            // 'is_active'                    => 'required|integer',
        ]);

        $complianceCode = $this->generateComplianceCode();

        $compliance = ComplianceMaster::create(array_merge($data, [
            'compliance_code' => $complianceCode,

        ]));


        return $this->successResponse(
            'Compliance created successfully'
        );
    }

    /**
     * Update compliance
     */
    public function updateCompliance(Request $request, $id)
    {
        $compliance = ComplianceMaster::where('id', $id)
            ->where('is_deleted', 0)
            ->first();

        if (!$compliance) {
            return $this->errorResponse('Compliance not found', null, 404);
        }

        $data = $request->validate([
            'compliance_month'          => 'required|string|max:20',
            'due_date'                  => 'required|date',
            'state_id'                  => 'required|exists:state_master,id',
            'statute_name'              => 'required|string|max:255',
            'form_name'                 => 'required|string|max:255',
            'nature_of_compliance_id'   => 'required|exists:nature_of_compliance_master,id',
            'submission_authority_id'   => 'required|exists:submission_authority_master,id',
            'is_active'                    => 'required|integer',
        ]);


        $compliance->update($data);

        return $this->successResponse(
            'Compliance updated successfully'
        );
    }

    /**
     * Soft delete compliance
     */
    public function deleteCompliance(Request $request, $id)
    {
        $compliance = ComplianceMaster::where('id', $id)
            ->where('is_deleted', 0)
            ->first();

        if (!$compliance) {
            return $this->errorResponse('Compliance not found', null, 404);
        }

        UserDeleteLog::create([
            'table_id' => $id,
            'table_name'  => 'compliance_master',
            'reason' => $request['reason'],
        ]);

        $compliance->update([
            'is_deleted' => 1
        ]);

        return $this->successResponse('Compliance deleted successfully');
    }

    private function generateComplianceCode()
    {
        do {
            // Generate 5-character alphanumeric code
            $code = strtoupper(Str::random(5));

            // Check if already exists
            $exists = ComplianceMaster::where('compliance_code', $code)->exists();
        } while ($exists); // If exists → repeat

        return $code;
    }

    public function createKeyAreaCompliance(Request $request)
    {
        $data = $request->validate([
            'name'          => 'required',
            'is_compliance'                  => 'required',
        ]);

        KeyAreasComplianceMaster::create($data);

        return $this->successResponse(
            'Compliance created successfully'
        );
    }

    /**
     * Update compliance
     */
    public function updateKeyAreaCompliance(Request $request, $id)
    {
        $KeyAreasComplianceMaster = KeyAreasComplianceMaster::where('id', $id)->first();

        if (!$KeyAreasComplianceMaster) {
            return $this->errorResponse('Key Areas Compliance not found', null, 404);
        }

        $data = $request->validate([
            'name'          => 'required',
            'is_compliance'  => 'required',

        ]);

        $KeyAreasComplianceMaster->update($data);

        return $this->successResponse(
            'Key Areas Compliance updated successfully'
        );
    }

    public function updateKeyAreaComplianceStatus(Request $request, $id)
    {
        $KeyAreasComplianceMaster = KeyAreasComplianceMaster::where('id', $id)->first();

        if (!$KeyAreasComplianceMaster) {
            return $this->errorResponse(
                'Key Areas Compliance not found',
                null,
                404
            );
        }

        $data = $request->validate([
            'is_active' => 'required|boolean',
        ]);

        $KeyAreasComplianceMaster->update([
            'is_active' => $data['is_active'],
        ]);

        return $this->successResponse(
            'Status updated successfully'
        );
    }


    public function deleteKeyAreaCompliance($id)
{
    $compliance = KeyAreasComplianceMaster::with('subCompliances')
        ->where('id', $id)
        ->first();

    if (!$compliance) {
        return $this->errorResponse(
            'Key Areas Compliance not found',
            null,
            404
        );
    }

    DB::beginTransaction();

    try {
        // ✅ Delete all related sub compliances
        $compliance->subCompliances()->delete();

        // ✅ Delete compliance
        $compliance->delete();

        DB::commit();

        return $this->successResponse(
            'Key Areas Compliance and Sub Compliance deleted successfully'
        );
    } catch (\Exception $e) {
        DB::rollBack();

        return $this->errorResponse(
            'Delete failed',
            $e->getMessage(),
            500
        );
    }
}

    public function listKeyAreaCompliance()
    {
        $keyAreasComplianceMaster = KeyAreasComplianceMaster::orderBy('name')->get();

        return $this->successResponse(
            'KeyAreas Compliance Master list fetched successfully',
            $keyAreasComplianceMaster
        );
    }

    /**
     * Create Sub Compliance
     */
    public function createKeyAreaSubCompliance(Request $request)
    {
        $data = $request->validate([
            'key_area_compliance_id' => 'required|exists:key_areas_compliance_master,id',
            'name'                  => 'required',
        ]);

        KeyAreaSubCompliance::create([
            'key_area_compliance_id' => $data['key_area_compliance_id'],
            'name'                   => $data['name'],
        ]);

        return $this->successResponse(
            'Sub Compliance created successfully'
        );
    }

    /**
     * Update Sub Compliance
     */
    public function updateKeyAreaSubCompliance(Request $request, $id)
    {
        $subCompliance = KeyAreaSubCompliance::where('id', $id)->first();

        if (!$subCompliance) {
            return $this->errorResponse(
                'Sub Compliance not found',
                null,
                404
            );
        }

        $data = $request->validate([
            'name' => 'required',
        ]);

        $subCompliance->update([
            'name' => $data['name'],
        ]);

        return $this->successResponse(
            'Sub Compliance updated successfully'
        );
    }

    /**
     * Update Sub Compliance Status
     */
    public function updateKeyAreaSubComplianceStatus(Request $request, $id)
    {
        $subCompliance = KeyAreaSubCompliance::where('id', $id)->first();

        if (!$subCompliance) {
            return $this->errorResponse(
                'Sub Compliance not found',
                null,
                404
            );
        }

        $data = $request->validate([
            'is_active' => 'required|boolean',
        ]);

        $subCompliance->update([
            'is_active' => $data['is_active'],
        ]);

        return $this->successResponse(
            'Status updated successfully'
        );
    }

    /**
     * Delete Sub Compliance
     */
    public function deleteKeyAreaSubCompliance($id)
    {
        $subCompliance = KeyAreaSubCompliance::where('id', $id)->first();

        if (!$subCompliance) {
            return $this->errorResponse(
                'Sub Compliance not found',
                null,
                404
            );
        }

        KeyAreaSubCompliance::where('id', $id)->delete();

        return $this->successResponse(
            'Sub Compliance deleted successfully'
        );
    }

    /**
     * List Sub Compliance by Key Area Compliance
     */
    public function listKeyAreaSubCompliance($keyAreaComplianceId)
    {
        $keyArea = KeyAreasComplianceMaster::where(
            'id',
            $keyAreaComplianceId
        )->first();

        if (!$keyArea) {
            return $this->errorResponse(
                'Key Areas Compliance not found',
                null,
                404
            );
        }

        $subCompliances = KeyAreaSubCompliance::where(
                'key_area_compliance_id',
                $keyAreaComplianceId
            )
            ->orderBy('name')
            ->get();

        return $this->successResponse(
            'Sub Compliance list fetched successfully',
            $subCompliances
        );
    }
}
